package main

import (
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"os"
	"os/signal"
	"path/filepath"
	"strings"
	"time"

	"github.com/howeyc/gopass"
	"github.com/urfave/cli/v2"
)

const (
	confFilePath = "/etc/apt/auth.conf.d/uos.conf"
	logFilePath  = "/tmp/uos-apt-login.log"
)

var (
	errUsernamePassword = errors.New("usename or password error")
	errNetwork          = errors.New("network error")
	errSystem           = errors.New("system error")
)

func main() {
	apiUrl := "https://apt-auth.chinauos.com/v1/apt/login"
	if os.Getenv("UOS_APT_LOGIN_URL") != "" {
		apiUrl = os.Getenv("UOS_APT_LOGIN_URL")
	}
	var (
		username string
		password string
	)
	app := &cli.App{
		Name:    "uos-apt-login",
		Usage:   "uos apt login",
		Version: "0.0.2",
		Commands: []*cli.Command{
			{
				Name:        "login",
				Aliases:     []string{"l"},
				Usage:       "uos apt login with flags",
				Description: "通过参数登录",
				Flags: []cli.Flag{
					&cli.StringFlag{Name: "username", Aliases: []string{"u"}, Destination: &username, Usage: "Username", Required: true},
					&cli.StringFlag{Name: "password", Aliases: []string{"p"}, Destination: &password, Usage: "Password", Required: true},
				},
				Action: func(c *cli.Context) error {
					if username == "" {
						return errors.New("username not empty")
					}
					if password == "" {
						return errors.New("password not empty")
					}
					p := param{
						Username: username,
						Password: password,
					}
					t, err := login(apiUrl, p)
					if err != nil {
						writeLog(err)
						return err
					}
					if t == "" {
						writeLog(err)
						return errUsernamePassword
					}
					if err := write(p.Username, t); err != nil {
						writeLog(err)
						return errSystem
					}
					return nil
				},
			},
		},
		Action: func(c *cli.Context) error {
			var (
				p param
			)
			handleInterrupt(os.Stdin.Fd())
			for {
				fmt.Printf("Username: ")
				n, err := fmt.Scan(&username)
				if err != nil {
					return err
				}
				if n == 0 {
					fmt.Println("请输入账号!")
					continue
				}
				p.Username = username
				break
			}
			for {
				fmt.Print("Password: ")
				password, err := gopass.GetPasswd()
				if err != nil {
					return err
				}
				if len(password) == 0 {
					fmt.Println("请输入密码!")
					continue
				}
				p.Password = string(password)
				break
			}

			t, err := login(apiUrl, p)
			if err != nil {
				writeLog(err)
				return err
			}
			if t == "" {
				writeLog(err)
				return errUsernamePassword
			}
			if err := write(p.Username, t); err != nil {
				writeLog(err)
				return errSystem
			}
			return nil
		},
	}
	err := app.Run(os.Args)
	if err != nil {
		writeLog(err)
	}
	switch err {
	case errUsernamePassword:
		os.Exit(1)
	case errNetwork:
		os.Exit(2)
	case errSystem:
		os.Exit(3)
	}
	os.Exit(0)
}

func writeLog(err error) {
	_ = os.MkdirAll(filepath.Dir(logFilePath), os.ModePerm)
	_ = os.Chmod(logFilePath, os.ModePerm)
	f, _ := os.OpenFile(logFilePath, os.O_CREATE|os.O_WRONLY|os.O_APPEND, 0666)
	_, _ = io.WriteString(f, "\r\n"+err.Error())
	defer f.Close()
	return
}

type param struct {
	Username string `json:"username"`
	Password string `json:"password"`
}

type responseValue struct {
	Token string `json:"token"`
}

type responseError struct {
	ErrorCn string `json:"error_cn"`
	ErrorEn string `json:"error_en"`
}

func login(u string, p param) (token string, err error) {
	var (
		val responseValue
		e   responseError
	)
	body, err := json.Marshal(p)
	if err != nil {
		return "", errSystem
	}

	req, err := http.NewRequest("POST", u, strings.NewReader(string(body)))
	if err != nil {
		return "", errSystem
	}

	client := http.Client{
		Timeout: time.Second * 30,
	}
	resp, err := client.Do(req)
	if err != nil {
		return "", errNetwork
	}
	defer resp.Body.Close()
	switch resp.StatusCode {
	case 200:
		r, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return "", errSystem
		}
		err = json.Unmarshal(r, &val)
		if err != nil {
			return "", errSystem
		}
		return val.Token, nil
	case 401:
		r, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return "", errSystem
		}
		err = json.Unmarshal(r, &e)
		if err != nil {
			return "", errSystem
		}
		fmt.Println(e.ErrorEn + "\n" + e.ErrorCn)
		return "", errUsernamePassword
	default:
		fmt.Println(resp.Status)
		return "", errNetwork
	}
}

func handleInterrupt(fd uintptr) {
	sigchan := make(chan os.Signal, 1)
	signal.Notify(sigchan, os.Interrupt)
	go func() {
		for range sigchan {
			// quit cleanly and the new terminal item is on a new line
			fmt.Println()
			signal.Stop(sigchan)
			close(sigchan)
			os.Exit(1)
		}
	}()
}

func write(username, token string) error {
	domain := []string{
		fmt.Sprintf("machine packages.chinauos.com login %s password %s", username, token),
		fmt.Sprintf("machine packages.chinauos.cn login %s password %s", username, token),
		fmt.Sprintf("machine uos.deepin.cn login %s password %s", username, token),
	}
	str := strings.Join(domain, "\n")
	content := []byte(str)
	_ = os.MkdirAll(filepath.Dir(confFilePath), 0755)

	return ioutil.WriteFile(confFilePath, content, 0644)
}
